<?php
// Include database connection
require_once 'conn.php';

// Check if required parameters are set
$required_params = ['token', 'amount', 'phone', 'network', 'Ported_number', 'airtime_type'];
$missing_params = [];

foreach ($required_params as $param) {
    if (!isset($_POST[$param])) {
        $missing_params[] = $param;
    }
}

if (!empty($missing_params)) {
    echo json_encode(["success" => false, "message" => "Missing parameters: " . implode(", ", $missing_params)]);
    exit();
}

// Assign POST data
$token = $_POST['token'];
$amount = $_POST['amount'];
$phone = $_POST['phone'];
$network = $_POST['network'];
$ported_number = $_POST['Ported_number'];
$airtime_type = $_POST['airtime_type'];
$service = "Airtime Purchase";

// Minimum amount check
if ($amount < 50) {
    echo json_encode(["success" => false, "message" => "Minimum airtime purchase amount is 50 Naira."]);
    exit();
}

try {
    // Step 1: Check user balance
    $user_query = "SELECT sId, sWallet FROM subscribers WHERE token = ?";
    $stmt = $conn->prepare($user_query);
    $stmt->bind_param("s", $token);
    $stmt->execute();
    $user_result = $stmt->get_result();

    if ($user_result->num_rows === 0) {
        echo json_encode(["success" => false, "message" => "Subscriber not found."]);
        exit();
    }

    $user_data = $user_result->fetch_assoc();
    $sId = $user_data['sId'];
    $user_balance = $user_data['sWallet'];

    if ($user_balance < $amount) {
        echo json_encode(["success" => false, "message" => "Insufficient balance."]);
        exit();
    }

    // Step 2: Get API details
    $api_query = "SELECT apikey, apilink FROM api2 WHERE value = 'airtime' LIMIT 1";
    $stmt = $conn->prepare($api_query);
    $stmt->execute();
    $api_result = $stmt->get_result();

    if ($api_result->num_rows === 0) {
        throw new Exception("API configuration not found.");
    }

    $api_data = $api_result->fetch_assoc();
    $api_url = $api_data['apilink'];
    $api_key = $api_data['apikey'];

    // Use different payloads based on API link
    if (in_array($api_url, ["https://nabatulusub.com/api/topup/", "https://legitdataway.com/api/topup/", "https://n3tdata247.com/api/topup/"])) {
        $payload = json_encode([
            'network' => $network,
            'phone' => $phone,
            'bypass' => false,
            'amount' => $amount,
            'plan_type' => 'VTU',
            'request-id' => uniqid("Data_")
        ]);
    } else {
        $payload = json_encode([
            'network' => $network,
            'amount' => $amount,
            'mobile_number' => $phone,
            'Ported_number' => true,
            'airtime_type' => 'VTU'
        ]);
    }

    // Step 3: Send API request using cURL
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Authorization: Token $api_key",
        "Content-Type: application/json"
    ]);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
    $response = curl_exec($ch);

    if (curl_errno($ch)) {
        throw new Exception("cURL error: " . curl_error($ch));
    }

    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    // DEBUG: Log full API response
    file_put_contents("airtime_debug.log", "HTTP Code: $http_code\nResponse: $response\n", FILE_APPEND);

    // Decode API response
    $response_data = json_decode($response, true);

    // Transaction reference
    $transref = uniqid("trans_", true);
    $timestamp = date("Y-m-d H:i:s");
    $description = "Airtime Purchase for $phone - $network";

    // Default status (failed)
    $status = 1;
    $new_balance = $user_balance; // Balance remains unchanged for failed transactions

    // Step 4: Check API response for success
    if ($http_code === 200 || $http_code === 201) {
        if (
            isset($response_data['status']) && 
            (strtolower($response_data['status']) === "successful" || 
             strtolower($response_data['Status']) === "successful" || 
             strtolower($response_data['status']) === "success")
        ) {
            // SUCCESSFUL TRANSACTION
            $status = 0;
            $new_balance = $user_balance - $amount;

            // Deduct balance
            $update_balance_query = "UPDATE subscribers SET sWallet = ? WHERE sId = ?";
            $stmt = $conn->prepare($update_balance_query);
            $stmt->bind_param("ds", $new_balance, $sId);
            if (!$stmt->execute()) {
                throw new Exception("Failed to update balance.");
            }

            // Send success response with API message
            $api_message = $response_data['api_response'] ?? "Transaction completed.";
            echo json_encode([
                "success" => true,
                "message" => "Airtime purchase successful.",
                "api_response" => $api_message
            ]);
        } else {
            // API responded but transaction failed
            $api_message = $response_data['api_response'] ?? "Transaction failed.";
            throw new Exception($api_message);
        }
    } else {
        // API request failed
        throw new Exception("API request failed with HTTP code $http_code.");
    }
} catch (Exception $e) {
    // FAILED TRANSACTION
    $status = 1;
    $description = "Failed: Airtime Purchase for $network - $phone";

    echo json_encode(["success" => false, "message" => $e->getMessage()]);
}

// Step 5: Insert transaction record (success or failure)
$transaction_query = "INSERT INTO transactions (sId, transref, servicename, servicedesc, amount, status, date, oldbal, newbal) 
                      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
$stmt = $conn->prepare($transaction_query);
$stmt->bind_param("isssdissd", $sId, $transref, $service, $description, $amount, $status, $timestamp, $user_balance, $new_balance);
$stmt->execute();

// Close database connection
$conn->close();
?>